#!/bin/bash
#  Copyright FUJITSU LIMITED 2025
####################################################
#
#
#
####################################################
set -euC
POSTGIS_WORK_DIR="$(pwd -P)"
OUT_DIR=$POSTGIS_WORK_DIR/OUT
BUILD_DIR=$POSTGIS_WORK_DIR/BUILD
SRC_DIR=$POSTGIS_WORK_DIR/SRC
LOG_DIR=$POSTGIS_WORK_DIR/LOG

POSTGIS_DIR=$SRC_DIR/postgis
CGAL_DIR=$SRC_DIR/cgal
SFCGAL_DIR=$SRC_DIR/sfcgal
PROJ_DIR=$SRC_DIR/proj
GDAL_DIR=$SRC_DIR/gdal
GEOS_DIR=$SRC_DIR/geos

FEP_VERSION=18
GISBS_VERSION="GISBSL-1835-01"
FEPSV_DIR="$(rpm -qa | grep -e FJSVfsep-server-core-${FEP_VERSION} | xargs rpm -ql 2>/dev/null | head -1)"

MAKE=make
CMAKE=cmake
PARA_T="all"

# Directory Structure
# - postgis-work
#     - postgis_build.sh
#     - postgis_test.sh
#     - SRC
#         - postgis
#         - cgal
#         - sfcgal
#         - proj
#         - gdal
#         - geos
#     - OUT
#         - postgis
#         - cgal
#         - sfcgal
#         - proj
#         - gdal
#         - geos
#     - BUILD
#     - LOG

function Usage()
{
    echo "$0 [Options]" 1>&2
    echo "  Options:" 1>&2
    echo "    -t|--target=OSS               Test specified OSS. OSS can be \"cgal\", \"sfcgal\", \"proj\", \"geos\", \"gdal\", or \"postgis\"" 1>&2
    echo "    -h|--help                     Display this help message and exit" 1>&2
    exit 2
}

function get_date() {
    date +'%Y/%m/%d %H:%M:%S'
}

function get_os_release()
{
    local os_release=/etc/os-release
    if [ "$(grep "SUSE Linux Enterprise" $os_release | wc -c)" -ne 0 ]; then
        OS_TYPE="SUSE"
    elif [ "$(grep "Red Hat Enterprise Linux" $os_release | wc -c)" -ne 0 ]; then
        OS_TYPE="RHEL"
    fi
    OS_MAJOR_VER="$(grep "VERSION_ID" $os_release | cut -f2 -d'"' | cut -f1 -d'.')"
}

function mv_test_dir(){
    if [ "${1}" == "postgis" ]; then
        cd $BUILD_DIR/$1
    else
        cd $BUILD_DIR/$1/build
    fi
}

function output_script_version()
{
    local oss=$1
    sed -i "1i----SCRIPT VERSION: ${GISBS_VERSION}----" "$LOG_DIR/${oss}/test-${oss}.log"
}

function test(){
    set -x

    target=$1
    mv_test_dir $target

    case $target in
        cgal)
            ctest -L CGAL_cmake_testsuite --no-tests=error && :
            result_errno=`expr $result_errno + $?`
        ;;
        sfcgal)
            ctest --no-tests=error && :
            result_errno=`expr $result_errno + $?`
        ;;
        proj)
            ctest --no-tests=error && :
            result_errno=`expr $result_errno + $?`
        ;;
        geos)
            bin/test_geos_unit && :
            result_errno=`expr $result_errno + $?`
        ;;
        gdal)
            if [ "${OS_TYPE}" == "RHEL" ] && [ "${OS_MAJOR_VER}" == "9" ]; then
                # test_vsizip_10 in autotest_gcore is skipped in this script
                sed -i -e 's/test_vsizip_10/vsizip_10/g' autotest/gcore/vsizip.py
            fi
            ctest --no-tests=error && :
            result_errno=`expr $result_errno + $?`
        ;;
        postgis)
            export PATH=$FEPSV_DIR/bin:${PATH:-}
            export LD_LIBRARY_PATH="$FEPSV_DIR/lib:$OUT_DIR/postgis/lib:${LD_LIBRARY_PATH:-}"
            ${MAKE} staged-install
            cp -rp ${OUT_DIR}/postgis/* ${BUILD_DIR}/postgis/regress/00-regress-install
            ln -s $FEPSV_DIR/lib/libpq.so.5 ${BUILD_DIR}/postgis/regress/00-regress-install/lib/libpq.so.5
            ${MAKE} installcheck && :
            result_errno=`expr $result_errno + $?`

            export LD_LIBRARY_PATH="$FEPSV_DIR/lib"
            cd extensions/address_standardizer
            ${MAKE} installcheck && :
            result_errno=`expr $result_errno + $?`

            mv_test_dir postgis
            cd extensions/postgis_tiger_geocoder
            # Test #1051d in pagc_normalize_address is skipped because there is a problem with test sets and it fails.
            # Please refer to the ticket: https://trac.osgeo.org/postgis/ticket/5408
            sed -i -e '/1051d/d' sql/test-pagc_normalize_address.sql
            sed -i -e '/1051d/d' expected/test-pagc_normalize_address.out

            ${MAKE} installcheck && :
            result_errno=`expr $result_errno + $?`
        ;;
        *)
            echo "unrecognized oss: ${target}."
            exit 1
        ;;
    esac
    set +x
    return $result_errno
}

# get option
while getopts :ht:-: opt ; do
    set +u
    optarg="${OPTARG-}"
    if [ "$opt" == - ]; then
        opt="-${OPTARG%%=*}"
        optarg="${OPTARG/${OPTARG%%=*}/}"
        optarg="${optarg#=}"
        if [ -z "$optarg" ] && [ ! "${!OPTIND}" = -* ]; then
            optarg="${!OPTIND}"
            shift
        fi
    fi
    set -u
    case "-$opt" in
        -t|--target)
            PARA_T="$optarg"
        ;;
        -h|--help)
            Usage
        ;;
        -:)
            echo "No OSS specified"
            echo "Refer to \"postgis_test.sh -h\" for more information"
            exit 1
        ;;
        -\?|--*)
            echo "Invalid option specified."
            echo "Refer to \"postgis_test.sh -h\" for more information"
            exit 1
        ;;
    esac
done
shift $((OPTIND - 1))

get_os_release
if [ "${PARA_T}" == "all"  ]; then
    target_OSS=("cgal" "sfcgal" "proj" "geos" "gdal" "postgis")
elif [ "${PARA_T}" == "" ] || [ "${PARA_T}" == "=" ]; then
    echo "No OSS specified"
    echo "Refer to \"postgis_test.sh -h\" for more information"
    exit 1
else
    target_OSS=("${PARA_T}")
fi

case $target_OSS in
    cgal|sfcgal|proj|geos|gdal|postgis)
    ;;
    *)
        echo "$(basename $0): Invalid OSS name specified"
        exit 1
    ;;
esac
for oss in ${target_OSS[@]}; do
    result_errno=0
    rm -rf $LOG_DIR/${oss}/test-${oss}.log
    echo -n "$(get_date) ${oss}: "
    test $oss $@ >| "$LOG_DIR/${oss}/test-${oss}.log" 2>&1 && :
    err=$?
    output_script_version ${oss}
    if [ ${err} -eq 0 ]; then
        echo "OK"
    else
        echo "NG --- Please check log files in ${LOG_DIR}/${oss}"
        result_errno=$err
    fi
done

cd $POSTGIS_WORK_DIR
get_date

exit ${result_errno}
