#!/bin/bash
#  Copyright FUJITSU LIMITED 2025
####################################################
#
#
#
####################################################
set -euC
POSTGIS_WORK_DIR="$(pwd -P)"
OUT_DIR=$POSTGIS_WORK_DIR/OUT
BUILD_DIR=$POSTGIS_WORK_DIR/BUILD
SRC_DIR=$POSTGIS_WORK_DIR/SRC
LOG_DIR=$POSTGIS_WORK_DIR/LOG

POSTGIS_DIR=$SRC_DIR/postgis
CGAL_DIR=$SRC_DIR/cgal
SFCGAL_DIR=$SRC_DIR/sfcgal
PROJ_DIR=$SRC_DIR/proj
GDAL_DIR=$SRC_DIR/gdal
GEOS_DIR=$SRC_DIR/geos

FEP_VERSION=18
GISBS_VERSION="GISBSL-1835-01"
FEPSV_DIR="$(rpm -qa | grep -e FJSVfsep-server-core-${FEP_VERSION} | xargs rpm -ql 2>/dev/null | head -1)"

MAKE=make
CMAKE=cmake

PARA_T="all"
WITH_LLVM=true

# Directory Structure
# - postgis-work
#     - postgis_build.sh
#     - postgis_test.sh
#     - SRC
#         - postgis
#         - cgal
#         - sfcgal
#         - proj
#         - gdal
#         - geos
#     - OUT
#         - postgis
#         - cgal
#         - sfcgal
#         - proj
#         - gdal
#         - geos
#     - BUILD
#         - postgis
#         - cgal
#         - sfcgal
#         - proj
#         - gdal
#         - geos
#     - LOG

function Usage()
{
    echo "$0 [Options]" 1>&2
    echo "  Options:" 1>&2
    echo "    -t|--target=OSS               Build specified OSS. OSS can be \"cgal\", \"sfcgal\", \"proj\", \"geos\", \"gdal\", or \"postgis\"" 1>&2
    echo "    --without-llvm                Build without support for LLVM based JIT compilation" 1>&2
    echo "    -h|--help                     Display this help message and exit" 1>&2
    exit 2
}

function get_date()
{
    date +'%Y/%m/%d %H:%M:%S'
}

function get_os_release()
{
    local os_release=/etc/os-release
    if [ "$(grep "SUSE Linux Enterprise" $os_release | wc -c)" -ne 0 ]; then
        OS_TYPE="SUSE"
    elif [ "$(grep "Red Hat Enterprise Linux" $os_release | wc -c)" -ne 0 ]; then
        OS_TYPE="RHEL"
    fi
    OS_MAJOR_VER="$(grep "VERSION_ID" $os_release | cut -f2 -d'"' | cut -f1 -d'.')"
}

function prepare_src()
{
    if [ ! -d $BUILD_DIR ]; then
        mkdir $BUILD_DIR
    fi
    cd $BUILD_DIR
    src=${1}
    SRC=${1^^}
    dir="${SRC}_DIR"
    rm -rf $src
    cp -rp ${!dir} ./
    cd $src
    find . -type f -exec sha256sum {} \; > ${LOG_DIR}/${oss}/hash-${oss}.txt
}

function output_log()
{
    cd $POSTGIS_WORK_DIR

    rm -f LOG/env.txt
    rm -f LOG/rpmfiles.txt
    rm -f ${OUT_DIR}/postgis/share/postgis-files.txt

    # If new files are added to the log directory, add them below.
    env > LOG/env.txt
    rpm -qa | sort > LOG/rpmfiles.txt
    rm -f ${OUT_DIR}/postgis/share/postgis-build-log.tar.gz
    tar zcf ${OUT_DIR}/postgis/share/postgis-build-log.tar.gz LOG
    cd ${OUT_DIR}/postgis
    # export all file and symbolic link name to postgis-files
    { find . -type f; find . -type l; } | sort > ${OUT_DIR}/postgis/share/postgis-files.txt
    sed -i -e "/postgis-files/d" ${OUT_DIR}/postgis/share/postgis-files.txt
}

function output_script_version()
{
    local oss=$1
    sed -i "1i----SCRIPT VERSION: ${GISBS_VERSION}----" "$LOG_DIR/${oss}/build-${oss}.log"
}

unnecessary_files=(\
    "garden-test-SFCGAL" \
    "standalone-regress-test-SFCGAL" \
    "test-regress-convex_hull" \
    "test-regress-polygon_triangulator" \
    "unit-test-SFCGAL" \
    )

function delete_file_in_directory()
{
    find "$1" -name "$2" | xargs rm -f
}

function delete_unnecessary_files()
{
    directory=$1
    shift
    for file in "$@"; do
        delete_file_in_directory "$directory" "$file"
    done
}

function build()
{
    set -x
    target=$1
    result_errno=0
    prepare_src $target

    case $target in
        cgal)
            mkdir ./build
            cd ./build
            mkdir -p "${OUT_DIR}/cgal"
            ${CMAKE} \
            -DCMAKE_BUILD_TYPE=Release \
            -DWITH_tests=ON \
            -DBUILD_TESTING=ON \
            -DCMAKE_INSTALL_LIBDIR=lib \
            -DCMAKE_INSTALL_PREFIX="${OUT_DIR}/cgal" \
            -DCMAKE_INSTALL_RPATH=\'\$ORIGIN/../lib\' \
            -DCMAKE_EXE_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DCMAKE_SHARED_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            .. && \
            ${MAKE} && :
            result_errno=`expr $result_errno + $?`
            ${MAKE} install
            # add for FEP 18
            # make test suite (5.6.0 <=)
            ${CMAKE} -DBUILD_TESTING=ON -DWITH_examples=ON -DWITH_tests=ON -DWITH_demos=ON ..
            cp -p CMakeCache.txt $LOG_DIR/cgal
        ;;
        sfcgal)
            mkdir ./build
            cd ./build
            mkdir -p "${OUT_DIR}/sfcgal"
            ${CMAKE} \
            -DCMAKE_BUILD_TYPE=Release \
            -DCMAKE_INSTALL_LIBDIR=lib \
            -DCGAL_DIR="${OUT_DIR}/cgal/lib/cmake/CGAL" \
            -DCMAKE_INSTALL_PREFIX="${OUT_DIR}/sfcgal" \
            -DCMAKE_INSTALL_RPATH=\'\$ORIGIN/../lib\' \
            -DCMAKE_EXE_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DCMAKE_SHARED_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DSFCGAL_BUILD_TESTS=ON \
            .. && \
            ${MAKE} -j1 && :
            result_errno=`expr $result_errno + $?`
            ${MAKE} install
            cp -p CMakeCache.txt $LOG_DIR/sfcgal
        ;;
        proj)
            mkdir ./build
            cd ./build
            mkdir -p ${OUT_DIR}/proj
            mkdir -p ${OUT_DIR}/proj/lib
            ${CMAKE} \
            -DCMAKE_BUILD_TYPE=Release \
            -DCMAKE_INSTALL_LIBDIR=lib \
            -DCMAKE_INSTALL_PREFIX="${OUT_DIR}/proj" \
            -DBUILD_TESTING=ON \
            -DCMAKE_INSTALL_RPATH=\'\$ORIGIN/../lib\' \
            -DCMAKE_EXE_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DCMAKE_SHARED_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DBUILD_PROJSYNC=OFF \
            -DENABLE_CURL=OFF \
            .. && \
            ${CMAKE} --build . && :
            result_errno=`expr $result_errno + $?`
            ${CMAKE} --build . --target install
            cp -p CMakeCache.txt $LOG_DIR/proj
        ;;
        geos)
            mkdir ./build
            cd ./build
            mkdir -p "${OUT_DIR}/geos"
            ${CMAKE} \
            -DCMAKE_BUILD_TYPE=Release \
            -DCMAKE_INSTALL_LIBDIR=lib \
            -DCMAKE_INSTALL_PREFIX="${OUT_DIR}/geos" \
            -DCMAKE_INSTALL_RPATH=\'\$ORIGIN/../lib\' \
            -DCMAKE_EXE_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DCMAKE_SHARED_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            .. && \
            ${MAKE} && :
            result_errno=`expr $result_errno + $?`
            ${MAKE} install
            cp -p CMakeCache.txt $LOG_DIR/geos
        ;;
        gdal)
            if [ "${OS_TYPE}" == "RHEL" ] && [ "${OS_MAJOR_VER}" == "9" ]; then
                python_bindings=ON
            	export LD_LIBRARY_PATH="$FEPSV_DIR/lib:${LD_LIBRARY_PATH:-}"
            else
                # turn off BUILD_PYTHON_BINDINGS:BOOL on RHEL8 and SUSE15 because 
                # the version of Python (especially pytest) does not meet the requirement
                python_bindings=OFF
            fi
            mkdir ./build
            cd ./build
            mkdir -p "${OUT_DIR}/gdal"
            ${CMAKE} \
            -DGDAL_USE_ARCHIVE=OFF \
            -DGDAL_USE_ARMADILLO=OFF \
            -DGDAL_USE_ARROW=OFF \
            -DGDAL_USE_BASISU=OFF \
            -DGDAL_USE_BLOSC=OFF \
            -DGDAL_USE_BRUNSLI=OFF \
            -DGDAL_USE_CFITSIO=OFF \
            -DGDAL_USE_CRNLIB=OFF \
            -DGDAL_USE_CURL=OFF \
            -DGDAL_USE_CRYPTOPP=OFF \
            -DGDAL_USE_DEFLATE=OFF \
            -DGDAL_USE_ECW=OFF \
            -DGDAL_USE_EXPAT=ON \
            -DGDAL_USE_FILEGDB=OFF \
            -DGDAL_USE_FREEXL=OFF \
            -DGDAL_USE_FYBA=OFF \
            -DGDAL_USE_GEOTIFF_INTERNAL=ON \
            -DGEOS_INCLUDE_DIR="${OUT_DIR}/geos/include" \
            -DGEOS_LIBRARY="${OUT_DIR}/geos/lib/libgeos_c.so" \
            -DGDAL_USE_GEOS=ON \
            -DGDAL_USE_GIF=ON \
            -DGDAL_USE_GIF_INTERNAL=OFF \
            -DGDAL_USE_GTA=OFF \
            -DGDAL_USE_HEIF=OFF \
            -DGDAL_USE_HDF4=OFF \
            -DGDAL_USE_HDF5=OFF \
            -DGDAL_USE_HDFS=OFF \
            -DGDAL_USE_ICONV=OFF \
            -DGDAL_USE_IDB=OFF \
            -DGDAL_USE_JPEG=ON \
            -DGDAL_USE_JPEG_INTERNAL=OFF \
            -DGDAL_USE_JPEG12_INTERNAL=ON \
            -DGDAL_USE_JSONC=ON \
            -DGDAL_USE_JSONC_INTERNAL=OFF \
            -DGDAL_USE_JXL=OFF \
            -DGDAL_USE_KDU=OFF \
            -DGDAL_USE_KEA=OFF \
            -DGDAL_USE_LERC=OFF \
            -DGDAL_USE_LERC_INTERNAL=OFF \
            -DGDAL_USE_LIBAEC=OFF \
            -DGDAL_USE_LIBKML=OFF \
            -DGDAL_USE_LIBLZMA=ON \
            -DGDAL_USE_LIBQB3=OFF \
            -DLIBXML2_LIBRARY="${FEPSV_DIR}/lib/libxml2.so.2" \
            -DGDAL_USE_LIBXML2=ON \
            -DGDAL_USE_LURATECH=OFF \
            -DGDAL_USE_LZ4=ON \
            -DGDAL_USE_MONGOCXX=OFF \
            -DGDAL_USE_MRSID=OFF \
            -DGDAL_USE_MSSQL_NCLI=OFF \
            -DGDAL_USE_MSSQL_ODBC=OFF \
            -DGDAL_USE_MYSQL=OFF \
            -DGDAL_USE_NETCDF=OFF \
            -DGDAL_USE_ODBC=ON \
            -DGDAL_USE_ODBCCPP=OFF \
            -DGDAL_USE_OGDI=OFF \
            -DGDAL_USE_OPENCAD_INTERNAL=ON \
            -DGDAL_USE_OPENCL=OFF \
            -DGDAL_USE_OPENEXR=OFF \
            -DGDAL_USE_OPENJPEG=OFF \
            -DGDAL_USE_OPENSSL=OFF \
            -DGDAL_USE_ORACLE=OFF \
            -DGDAL_USE_PARQUET=OFF \
            -DGDAL_USE_PCRE2=ON \
            -DGDAL_USE_PDFIUM=OFF \
            -DGDAL_USE_PNG=ON \
            -DGDAL_USE_POPPLER=ON \
            -DPostgreSQL_INCLUDE_DIR="${FEPSV_DIR}/include" \
            -DPostgreSQL_LIBRARY_RELEASE="${FEPSV_DIR}/lib/libpq.so" \
            -DGDAL_USE_POSTGRESQL=ON \
            -DPROJ_INCLUDE_DIR="${OUT_DIR}/proj/include" \
            -DPROJ_LIBRARY_RELEASE="${OUT_DIR}/proj/lib/libproj.so" \
            -DGDAL_USE_QHULL_INTERNAL=ON \
            -DGDAL_USE_RASTERLITE2=OFF \
            -DGDAL_USE_RDB=OFF \
            -DGDAL_USE_SPATIALITE=OFF \
            -DGDAL_USE_SQLITE3=ON \
            -DSFCGAL_INCLUDE_DIR="${OUT_DIR}/sfcgal/include" \
            -DSFCGAL_LIBRARY_RELEASE="${OUT_DIR}/sfcgal/lib/libSFCGAL.so" \
            -DGDAL_USE_SFCGAL=ON \
            -DGDAL_USE_TEIGHA=OFF \
            -DGDAL_USE_TIFF=ON \
            -DGDAL_USE_TIFF_INTERNAL=OFF \
            -DGDAL_USE_TILEDB=OFF \
            -DGDAL_USE_WEBP=ON \
            -DGDAL_USE_XERCESC=OFF \
            -DGDAL_USE_ZLIB=ON \
            -DGDAL_USE_ZLIB_INTERNAL=OFF \
            -DGDAL_USE_ZSTD=ON \
            -DBUILD_PYTHON_BINDINGS:BOOL=${python_bindings} \
            -DBUILD_JAVA_BINDINGS:BOOL=ON \
            -DGDAL_JAVA_GENERATE_JAVADOC:BOOL=OFF \
            -DCMAKE_INSTALL_LIBDIR=lib \
            -DCMAKE_INSTALL_PREFIX="${OUT_DIR}/gdal" \
            -DCMAKE_INSTALL_RPATH=\'\$ORIGIN/../lib\' \
            -DCMAKE_EXE_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DCMAKE_SHARED_LINKER_FLAGS="-Wl,--enable-new-dtags" \
            -DCMAKE_BUILD_TYPE=Release \
            .. && \
            ${CMAKE} --build . && :
            result_errno=`expr $result_errno + $?`
            ${CMAKE} --build . --target install
            cp -rp ${OUT_DIR}/gdal/lib64/* ${OUT_DIR}/gdal/lib && rm -rf ${OUT_DIR}/gdal/lib64
            cp -p CMakeCache.txt $LOG_DIR/gdal
        ;;
        postgis)
            export LD_LIBRARY_PATH="$FEPSV_DIR/lib:$OUT_DIR/proj/lib:$OUT_DIR/gdal/lib:$OUT_DIR/geos/lib:$OUT_DIR/sfcgal/lib:${LD_LIBRARY_PATH:-}"
            pg_config_bin="${FEPSV_DIR}/bin/pg_config"
            PKG_CONFIG_PATH="${OUT_DIR}/proj/lib/pkgconfig" \
            ./configure \
                --prefix="${OUT_DIR}/postgis" \
                --with-pgconfig="${pg_config_bin}" \
                --with-gdalconfig="${OUT_DIR}/gdal/bin/gdal-config" \
                --with-geosconfig="${OUT_DIR}/geos/bin/geos-config" \
                --with-sfcgal="${OUT_DIR}/sfcgal/bin/sfcgal-config" \
                LDFLAGS="-Wl,-rpath,'\$\$ORIGIN/../lib',--enable-new-dtags" && :
            result_errno=`expr $result_errno + $?`
            if "${WITH_LLVM}"; then
                ${MAKE} USE_PGXS=1 PG_CONFIG="${pg_config_bin}" && :
                result_errno=`expr $result_errno + $?`
                ${MAKE} USE_PGXS=1 PG_CONFIG="${pg_config_bin}" install DESTDIR=$OUT_DIR/postgis/tmp && :
                result_errno=`expr $result_errno + $?`
            else
                ${MAKE} USE_PGXS=1 PG_CONFIG="${pg_config_bin}" with_llvm=no && :
                result_errno=`expr $result_errno + $?`
                ${MAKE} USE_PGXS=1 PG_CONFIG="${pg_config_bin}" install DESTDIR=$OUT_DIR/postgis/tmp with_llvm=no && :
                result_errno=`expr $result_errno + $?`
            fi
            cp -pr $OUT_DIR/postgis/tmp$FEPSV_DIR/* $OUT_DIR/postgis && cp -pr $OUT_DIR/postgis/tmp$OUT_DIR/postgis/* $OUT_DIR/postgis && :
            result_errno=`expr $result_errno + $?`
            rm -rf $OUT_DIR/postgis/tmp && :
            result_errno=`expr $result_errno + $?`

            cp -rp ${OUT_DIR}/proj/* ${OUT_DIR}/postgis && cp -rp ${OUT_DIR}/gdal/* ${OUT_DIR}/postgis && cp -rp ${OUT_DIR}/geos/* ${OUT_DIR}/postgis && cp -rp ${OUT_DIR}/sfcgal/* ${OUT_DIR}/postgis && :
            result_errno=`expr $result_errno + $?`
            delete_unnecessary_files $OUT_DIR/postgis "${unnecessary_files[@]}"
            mkdir -p ${OUT_DIR}/postgis/share/postgis-utils && cp -rp ${BUILD_DIR}/postgis/utils/*.pl ${OUT_DIR}/postgis/share/postgis-utils && cp -rp ${BUILD_DIR}/postgis/utils/README ${OUT_DIR}/postgis/share/postgis-utils && :
            result_errno=`expr $result_errno + $?`
            cp -p config.log $LOG_DIR/postgis && :
            result_errno=`expr $result_errno + $?`

            output_log && :
            result_errno=`expr $result_errno + $?`
        ;;
        *)
            echo "unrecognized oss: ${target}."
            exit 1
        ;;
    esac
    set +x
    return $result_errno
}

# get option
while getopts :ht:-: opt; do
    set +u
    optarg="${OPTARG-}"
    if [ "$opt" == - ]; then
        opt="-${OPTARG%%=*}"
        optarg="${OPTARG/${OPTARG%%=*}/}"
        optarg="${optarg#=}"
        if [ -z "$optarg" ] && [ ! "${!OPTIND}" = -* ]; then
            optarg="${!OPTIND}"
            shift
        fi
    fi
    set -u
    case "-$opt" in
        -t|--target) 
            PARA_T="$optarg"
        ;;
        -h|--help)
            Usage
        ;;
        --without-llvm)
            WITH_LLVM=false
        ;;
        -:)
            echo "No OSS specified."
            echo "Refer to \"postgis_build.sh -h\" for more information"
            exit 1
        ;;
        -\?|--*) 
            echo "Invalid option specified."
            echo "Refer to \"postgis_build.sh -h\" for more information"
            exit 1
        ;;
    esac
done
shift $((OPTIND - 1))

get_os_release
if [ "${PARA_T}" == "all" ]; then
    target_OSS=("cgal" "sfcgal" "proj" "geos" "gdal" "postgis")
elif [ "${PARA_T}" == "" ] || [ "${PARA_T}" == "=" ]; then
    echo "No OSS specified."
    echo "Refer to \"postgis_build.sh -h\" for more information"
    exit 1
else
    target_OSS=("${PARA_T}")
fi

case $target_OSS in
    cgal|sfcgal|proj|geos|gdal|postgis)
    ;;
    *)
        echo "$(basename $0): Invalid OSS name specified"
        exit 1
    ;;
esac
for oss in ${target_OSS[@]}; do
    result_errno=0
    rm -rf $LOG_DIR/${oss}
    mkdir -p $LOG_DIR/${oss}
    rm -rf $OUT_DIR/${oss}
    mkdir -p $OUT_DIR/${oss}
    echo -n "$(get_date) ${oss}: "
    build $oss $@ >| "$LOG_DIR/${oss}/build-${oss}.log" 2>&1 && :
    err=$?
    output_script_version ${oss}
    if [ ${err} -eq 0 ]; then
        echo "OK"
    else
        echo "NG --- Please check log files in ${LOG_DIR}/${oss}"
        result_errno=$err
	exit ${result_errno}
    fi
done

get_date

exit ${result_errno}
